% Simulation of Correlated 1-D Random Walks (RWs)
% by Jorge E. Pezoa
% Approach: the script simulates the RW using a Markov Chain for the
% direction of the jump. A jump to the left (-1) has associated a
% probability of changing the direction (PrChangeLeft) or keep jumping to
% the left  (1-PrChangeLeft). The same model is applied for the jumps to
% the right (PrChangeRight and 1-PrChangeRight).
% Using such Markov Chain, the position at each step (or jump) is computed.
% The jumps can also be simulated deterministic or random and discrete or
% continuous.

clear all; clc

% Simulation parameters
MaxN=100;                        % Max number of time steps
MaxRealizations=50000;           % Max number of realizations to simulate
Paths2Plot=400;                  % Maximum number of path to plot on the Lattice
Dir=zeros(MaxRealizations,MaxN+1); % Allocate memory space
Pos=zeros(MaxRealizations,MaxN+1);
NJumps=5; % The number of the last NJumps jumps for which
          % the correlation coefficient is computed
JumpType=0; % The type of jump:
            % 0: deterministic
            % 1: discrete uniformly distributed 
            % 2: random exponentially distributed
% Define the parameters (statistics) of the jump size
switch JumpType
    case 0
        JumpStats=2; % The size of the each jump
    case 1
        JumpStats=4; % The maximum jump size for discret uniformly distributed jumps
    case 2
        JumpStats=2; %The mean the jump size for exponentially distributed jumps
end
% Prob. of changing the direction of movement when the direction of the previous jump was Left
PrChangeLeft =0.20; 
% Prob. of changing the direction of movement when the direction of the previous jump was Right
PrChangeRight=0.30; 

% Randomdly choose the initial direction. By convenience it is the same for
% all the simulations
RandN=rand;
Dir0=(RandN<0.5)-(RandN>=0.5); % selects the initial direction
Dir(:,1)=Dir0;  % Initial direction: computed randomly (for all the realizations)
Pos(:,1)=Dir0;  % Initial position: assumed 0 at time 0 (for all the realizations)

for i=1:MaxRealizations % Loop for the number of realizations (in rows)
    for j=2:MaxN+1        % Loop for the number of jumps (in columns)
        % Check if the previous direction was to left (-1)
        if (Dir(i,j-1)==-1)
            % Change direction according to prob. of jump to left
            if (rand<PrChangeLeft)
                Dir(i,j)=-Dir(i,j-1);
                % Keep jumping to the left
            else
                Dir(i,j)=Dir(i,j-1);
            end
            % Check if the previous direction was to right (+1)
        else
            % Change direction according to prob. of jump to right
            if (rand<PrChangeRight)
                Dir(i,j)=-Dir(i,j-1);
                % Keep jumping to the right
            else
                Dir(i,j)=Dir(i,j-1);
            end
        end
        % Calculate the new position
        % NOTE: IF YOU WANT TO SIMULATE DETERMINISTIC JUMPS JUST COMMENT
        % THE FOLLOWING LINES AND UNCOMMENT THE LAST LINE OF THE LOOP TO
        % SPEED-UP THE CODE
        % COMMENT or UNCOMMENT FROM HERE
        switch JumpType
            case 0 % Deterministic and discrete jump
                Size=JumpStats(1);
            case 1 % Random jump Gaussian distributed
                Size=round(rand*JumpStats(1));
            case 2 % Random jump Exponentially distributed
                Size=-(1/JumpStats(1))*log(rand);
        end
        Pos(i,j)=Pos(i,j-1)+Dir(i,j)*Size;
        % COMMENT or UNCOMMENT UNTIL HERE
        %
        % UNCOMMENT THIS LAST LINE TO SPEED-UP DETERMISTIC SIMULATIONS
        % (COMMENT THE PREVIOUS LINES)
        % Pos(i,j)=Pos(i,j-1)+Dir(i,j)*JumpStats(1);
    end
end

kf=1;
figure(kf); kf=kf+1;
Paths=fix(linspace(1,MaxRealizations,Paths2Plot));
plot((1:MaxN+1)-1,mean(Pos,1),'r-x','LineWidth',1);
hold on
plot((1:MaxN+1)-1,var(Pos,1)/5,'b-o','LineWidth',1);
plot(0:MaxN,Pos(Paths,:)');
hold off;
xlabel('k','FontName','Times','FontSize',16);
ylabel('Position[k]','FontName','Times','FontSize',16);
legend('Mean Disp.','MS Disp.','Lattice','Location','NorthWest')
title({'Lattice of a 1-D','Correlated--Random Walk'},'FontName','Times','FontSize',18);

% Plot the histogram of the direction of jumping
Counts=sum(hist(Dir',[-1 1]),2);
figure(kf); kf=kf+1;
freq=Counts/sum(Counts)
% Compute the theoretical values
A=[1-PrChangeLeft PrChangeRight; PrChangeLeft 1-PrChangeRight];
A=eye(2)-A; A=[1 1; A]; b=[1;0];
SteadyStatePr=A(1:2,1:2)\b
bar([-1 1],freq,0.25); hold on;
stem([-1 1], SteadyStatePr,'r'); hold off;
legend('Estim. Prob.','Theo. Prob.','Location','Best')
xlabel('k','FontName','Times','FontSize',16);
ylabel('Direction[k]','FontName','Times','FontSize',16);
title('Steady state pmf of the direction of the jumps','FontName','Times','FontSize',18);

if (JumpType==0 || JumpType==1)
    % Plot the histogram of the position at the (MaxN-1)-th jump
    Min=min(Pos(:,MaxN)); Max=max(Pos(:,MaxN));
    freq=hist(Pos(:,MaxN),Min:Max)/MaxRealizations;
    figure(kf); kf=kf+1;
    stem(Min:Max,freq);
    xlabel('x','FontName','Times','FontSize',16);
    ylabel('Position_{MaxN-1}[x]','FontName','Times','FontSize',16);
    title('pmf of the position at the (MaxN-1)-th jump','FontName','Times','FontSize',18);

    % Plot the histogram of the position at the MaxN-th jump
    Min=min(Pos(:,MaxN+1)); Max=max(Pos(:,MaxN+1));
    freq=hist(Pos(:,MaxN+1),Min:Max)/MaxRealizations;
    figure(kf); kf=kf+1;
    stem(Min:Max,freq);
    xlabel('x','FontName','Times','FontSize',16);
    ylabel('Position_{MaxN}[x]','FontName','Times','FontSize',16);
    title('pmf of the position at the (MaxN)-th jump','FontName','Times','FontSize',18);
end

% Compute the correlation coefficient between the last NJumps
NJumps=5;       % Number of jumps to compute the correlation coefficient
corrcoef(Pos(:,(end-NJumps):end))